function [ lbar ] = charPathLength(A)
%
% Determine the chacteristic path length of a graph
% with adjacency matrix A, a_ij \in {0,1}
%   Results:
%    if lbar=0, adjacency matrix is not quadratic
%    if lbar=N, graph is not connectd
%
%    characteristic path length = average length of the shortest paths
%    among all vertices (i,j), excluding the paths i -> i of length 0;
%    if the graph is not strongly connected, lbar=0
%    function works for directed and undirected graphs
%  
% J. Lunze
% 
% 02.06.2018
% Version v. 20.6.2018
%
[N,M]=size(A);
if N ~= M
   lbar=0;             % if A is not quadratic, lbar=0
   return
end
%
N=length(A);
A=double(abs(A)>0);   %  to get matrix with elements in {0,1}
%
%  Method:
%  Determine A^k to find which vertices are connected by a path of length k
%  Length - matrix with l_ij = length of shortest path j -> i
%        initially, Length=N*ones(N,N),
%        hence, each element l_ij < N indicates that a path j -> i has
%        already been found
%        if finally l_ij=N exists, the graph is not connected
%  Ak - A^k
%  k - length of currently found paths
%
Length=N*ones(N,N); % initialisation
Length=Length-diag(diag(Length)); % introduce zero-length paths P(i-i)
%
Ak=eye(N,N);
Nones=N*ones(N,N);
k=0;
findmore=1;
while findmore == 1 && k < N
   Ak=Ak*A; 
   Ak=double(abs(Ak)>0);     % Normalisation to get elements in {0,1}
   k=k+1;        % current path length
   Nonconnected=Nones-N*Ak;   
   Nonconnected=Nonconnected+k*Ak;   
                                 % elements=k show new paths
   Length=min(Length,Nonconnected);   % introduce new shortest paths
   if max(max(Length))~=N
       findmore=0;                 % all paths found
   end
end
if max(max(Length))==N
    lbar=N;                      % graph is not connected
else
    lsum=sum(sum(Length));       % sum up all pathlengths
    lbar = lsum/N/(N-1);
end

